<?php

namespace StripeIntegration\Tax\Model\Total\Invoice;

use \Magento\Sales\Model\Order\Total\AbstractTotal;
use StripeIntegration\Tax\Model\StripeTax;
use \Magento\Sales\Model\Order\Invoice;

class Tax extends AbstractTotal
{
    private $stripeTax;
    private $invoiceHelper;

    public function __construct(
        StripeTax $stripeTax,
        \StripeIntegration\Tax\Helper\Invoice $invoiceHelper,
        array $data = [])
    {
        parent::__construct($data);

        $this->stripeTax = $stripeTax;
        $this->invoiceHelper = $invoiceHelper;
    }

    /**
     * @param Invoice $invoice
     * @return $this
     *
     * By giving this collector a big number, we want it to be executed last. This is because in case there are changes
     * to the tax rates, the prices which are set might be negative if we don't set them last.
     */
    public function collect(Invoice $invoice)
    {
        if ($this->stripeTax->isEnabled() && $this->stripeTax->hasValidResponse()) {
            $this->setDataFromStripeCalculatedValues($invoice);
        }

        return $this;
    }

    public function setDataFromStripeCalculatedValues($invoice)
    {
        $totalTax = 0;
        $baseTotalTax = 0;
        $totalDiscountTaxCompensation = 0;
        $baseTotalDiscountTaxCompensation = 0;
        $subtotal = 0;
        $baseSubtotal = 0;
        $subtotalInclTax = 0;
        $baseSubtotalInclTax = 0;

        foreach ($invoice->getAllItems() as $item) {
            $orderItem = $item->getOrderItem();
            $orderItemQty = $orderItem->getQtyOrdered();

            if ($orderItemQty) {
                if ($item->getOrderItem()->isDummy() || $item->getQty() <= 0) {
                    continue;
                }
                // get values set during tax calculation
                $calculatedValues = $item->getStripeCalculatedValues();
                $baseCalculatedValues = $item->getStripeBaseCalculatedValues();

                $this->removeInitialItemValues($item);

                // use the current values of the field and add to it, as some other collectors may have added to it
                $item->setTaxAmount($item->getTaxAmount() + $calculatedValues['row_tax']);
                $item->setBaseTaxAmount($item->getBaseTaxAmount() + $baseCalculatedValues['row_tax']);
                $item->setDiscountTaxCompensationAmount($item->getDiscountTaxCompensationAmount() + $calculatedValues['discount_tax_compensation']);
                $item->setBaseDiscountTaxCompensationAmount($item->getBaseDiscountTaxCompensationAmount() + $baseCalculatedValues['discount_tax_compensation']);

                $this->setItemAdditionalData($item, $calculatedValues, $baseCalculatedValues);

                $subtotal += $calculatedValues['row_total'];
                $baseSubtotal += $baseCalculatedValues['row_total'];
                $subtotalInclTax += $calculatedValues['row_total_incl_tax'];
                $baseSubtotalInclTax += $baseCalculatedValues['row_total_incl_tax'];

                $totalTax += $calculatedValues['row_tax'];
                $baseTotalTax += $baseCalculatedValues['row_tax'];
                $totalDiscountTaxCompensation += $calculatedValues['discount_tax_compensation'];
                $baseTotalDiscountTaxCompensation += $baseCalculatedValues['discount_tax_compensation'];
            }
        }

        $this->removeInitialInvoiceValues($invoice);
        $this->setInvoiceAdditionalData($invoice, $subtotal, $baseSubtotal, $subtotalInclTax, $baseSubtotalInclTax);

        $taxDiscountCompensationAmt = $totalDiscountTaxCompensation;
        $baseTaxDiscountCompensationAmt = $baseTotalDiscountTaxCompensation;

        if ($this->invoiceHelper->canIncludeShipping($invoice)) {
            $calculatedShippingValues = $invoice->getStripeCalculatedShippingValues();
            $baseCalculatedShippingValues = $invoice->getStripeBaseCalculatedShippingValues();

            $totalTax += $calculatedShippingValues['row_tax'];
            $baseTotalTax += $baseCalculatedShippingValues['row_tax'];
            $totalDiscountTaxCompensation += $calculatedShippingValues['discount_tax_compensation'];
            $baseTotalDiscountTaxCompensation += $baseCalculatedShippingValues['discount_tax_compensation'];

            $this->setAdditionalInvoiceShippingData($invoice, $calculatedShippingValues, $baseCalculatedShippingValues);

            $invoice->setShippingTaxAmount($invoice->getShippingTaxAmount() + $calculatedShippingValues['row_tax']);
            $invoice->setBaseShippingTaxAmount($invoice->getBaseShippingTaxAmount() + $baseCalculatedShippingValues['row_tax']);
            $invoice->setShippingDiscountTaxCompensationAmount($invoice->getShippingDiscountTaxCompensationAmount() + $calculatedShippingValues['discount_tax_compensation']);
            $invoice->setBaseShippingDiscountTaxCompensationAmnt($invoice->getBaseShippingDiscountTaxCompensationAmnt() + $baseCalculatedShippingValues['discount_tax_compensation']);
        }

        $invoice->setTaxAmount($invoice->getTaxAmount() + $totalTax);
        $invoice->setBaseTaxAmount($invoice->getBaseTaxAmount() + $baseTotalTax);
        $invoice->setDiscountTaxCompensationAmount($invoice->getDiscountTaxCompensationAmount() + $taxDiscountCompensationAmt);
        $invoice->setBaseDiscountTaxCompensationAmount($invoice->getBaseDiscountTaxCompensationAmount() + $baseTaxDiscountCompensationAmt);

        $invoice->setGrandTotal(
            $invoice->getGrandTotal() + $totalTax + $totalDiscountTaxCompensation + $invoice->getSubtotal() + $invoice->getShippingAmount()
        );
        $invoice->setBaseGrandTotal(
            $invoice->getBaseGrandTotal() + $baseTotalTax + $baseTotalDiscountTaxCompensation + $invoice->getBaseSubtotal() + $invoice->getBaseShippingAmount()
        );

        $this->addGwTaxes($invoice);
    }

    /**
     * @param $item
     * @return void
     *
     * Subtract the initial values of the item so that we can use the newly calculated values
     */
    private function removeInitialItemValues($item)
    {
        $item->setTaxAmount($item->getTaxAmount() - $item->getInitialTaxAmount());
        $item->setBaseTaxAmount($item->getBaseTaxAmount() - $item->getInitialBaseTaxAmount());
        $item->setDiscountTaxCompensationAmount($item->getDiscountTaxCompensationAmount() - $item->getInitialDiscountTaxCompensationAmount());
        $item->setBaseDiscountTaxCompensationAmount($item->getBaseDiscountTaxCompensationAmount() - $item->getInitialBaseDiscountTaxCompensationAmount());

        $item->setPrice($item->getPrice() - $item->getInitialPrice());
        $item->setBasePrice($item->getBasePrice() - $item->getInitialBasePrice());
        $item->setPriceInclTax($item->getPriceInclTax() - $item->getInitialPriceInclTax());
        $item->setBasePriceInclTax($item->getBasePriceInclTax() - $item->getInitialBasePriceInclTax());
        $item->setRowTotal($item->getRowTotal() - $item->getInitialRowTotal());
        $item->setBaseRowTotal($item->getBaseRowTotal() - $item->getInitialBaseRowTotal());
        $item->setRowTotalInclTax($item->getRowTotalInclTax() - $item->getInitialRowTotalInclTax());
        $item->setBaseRowTotalInclTax($item->getBaseRowTotalInclTax() - $item->getInitialBaseRowTotalInclTax());
    }

    /**
     * @param $invoice
     * @return void
     *
     * Subtract the initial values of the invoice so that we can use the newly calculated values
     */
    private function removeInitialInvoiceValues($invoice)
    {
        // For the Grand Total we take out the initial subtotal, shipping amount and tax and discount compensations
        $invoice->setGrandTotal($invoice->getGrandTotal() -
            $invoice->getInitialSubtotal() -
            $invoice->getInitialShippingAmount() -
            $invoice->getInitialTaxAmount() -
            $invoice->getInitialShippingDiscountTaxCompensationAmount() -
            $invoice->getInitialDiscountTaxCompensationAmount()
        );
        $invoice->setBaseGrandTotal($invoice->getBaseGrandTotal() -
            $invoice->getInitialBaseSubtotal() -
            $invoice->getInitialBaseShippingAmount() -
            $invoice->getInitialBaseTaxAmount() -
            $invoice->getInitialBaseShippingDiscountTaxCompensationAmnt() -
            $invoice->getInitialBaseDiscountTaxCompensationAmount()
        );

        $invoice->setShippingTaxAmount($invoice->getShippingTaxAmount() - $invoice->getInitialShippingTaxAmount());
        $invoice->setBaseShippingTaxAmount($invoice->getBaseShippingTaxAmount() - $invoice->getInitialBaseShippingTaxAmount());
        $invoice->setShippingDiscountTaxCompensationAmount($invoice->getShippingDiscountTaxCompensationAmount() - $invoice->getInitialShippingDiscountTaxCompensationAmount());
        $invoice->setBaseShippingDiscountTaxCompensationAmnt($invoice->getBaseShippingDiscountTaxCompensationAmnt() - $invoice->getInitialBaseShippingDiscountTaxCompensationAmnt());
        $invoice->setTaxAmount($invoice->getTaxAmount() - $invoice->getInitialTaxAmount());
        $invoice->setBaseTaxAmount($invoice->getBaseTaxAmount() - $invoice->getInitialBaseTaxAmount());
        $invoice->setDiscountTaxCompensationAmount($invoice->getDiscountTaxCompensationAmount() - $invoice->getInitialDiscountTaxCompensationAmount());
        $invoice->setBaseDiscountTaxCompensationAmount($invoice->getBaseDiscountTaxCompensationAmount() - $invoice->getInitialBaseDiscountTaxCompensationAmount());

        $invoice->setSubtotal($invoice->getSubtotal() - $invoice->getInitialSubtotal());
        $invoice->setBaseSubtotal($invoice->getBaseSubtotal() - $invoice->getInitialBaseSubtotal());
        $invoice->setSubtotalInclTax($invoice->getSubtotalInclTax() - $invoice->getInitialSubtotalInclTax());
        $invoice->setBaseSubtotalInclTax($invoice->getBaseSubtotalInclTax() - $invoice->getInitialBaseSubtotalInclTax());

        $invoice->setShippingAmount($invoice->getShippingAmount() - $invoice->getInitialShippingAmount());
        $invoice->setBaseShippingAmount($invoice->getBaseShippingAmount() - $invoice->getInitialBaseShippingAmount());
        $invoice->setShippingInclTax($invoice->getShippingInclTax() - $invoice->getInitialShippingInclTax());
        $invoice->setBaseShippingInclTax($invoice->getBaseShippingInclTax() - $invoice->getInitialBaseShippingInclTax());
    }

    /**
     * Sets the values which may change in case the tax rate changes between the point of order creation and the
     * point of invoice creation for invoice items
     */
    private function setItemAdditionalData($item, $calculatedValues, $baseCalculatedValues)
    {
        // use current value and add to it
        $item->setPrice($item->getPrice() + $calculatedValues['price']);
        $item->setBasePrice($item->getBasePrice() + $baseCalculatedValues['price']);
        $item->setPriceInclTax($item->getPriceInclTax() + $calculatedValues['price_incl_tax']);
        $item->setBasePriceInclTax($item->getBasePriceInclTax() + $baseCalculatedValues['price_incl_tax']);
        $item->setRowTotal($item->getRowTotal() + $calculatedValues['row_total']);
        $item->setBaseRowTotal($item->getBaseRowTotal() + $baseCalculatedValues['row_total']);
        $item->setRowTotalInclTax($item->getRowTotalInclTax() + $calculatedValues['row_total_incl_tax']);
        $item->setBaseRowTotalInclTax($item->getBaseRowTotalInclTax() + $baseCalculatedValues['row_total_incl_tax']);
    }

    /**
     * Sets invoice shipping related values which may change in case the tax rate changes between the point of order creation
     * and the point of invoice creation
     */
    private function setAdditionalInvoiceShippingData($invoice, $calculatedShippingValues, $baseCalculatedShippingValues)
    {
        // use current value and add to it
        $invoice->setShippingAmount($invoice->getShippingAmount() + $calculatedShippingValues['row_total']);
        $invoice->setBaseShippingAmount($invoice->getBaseShippingAmount() + $baseCalculatedShippingValues['row_total']);
        $invoice->setShippingInclTax($invoice->getShippingInclTax() + $calculatedShippingValues['row_total_incl_tax']);
        $invoice->setBaseShippingInclTax($invoice->getBaseShippingInclTax() + $baseCalculatedShippingValues['row_total_incl_tax']);
    }

    /**
     * Sets invoice related values which may change in case the tax rate changes between the point of order creation
     * and the point of invoice creation
     */
    private function setInvoiceAdditionalData($invoice, $subtotal, $baseSubtotal, $subtotalInclTax, $baseSubtotalInclTax)
    {
        // use current value and add to it
        $invoice->setSubtotal($invoice->getSubtotal() + $subtotal);
        $invoice->setBaseSubtotal($invoice->getBaseSubtotal() + $baseSubtotal);
        $invoice->setSubtotalInclTax($invoice->getSubtotalInclTax() + $subtotalInclTax);
        $invoice->setBaseSubtotalInclTax($invoice->getBaseSubtotalInclTax() + $baseSubtotalInclTax);
    }

    /**
     * @param $invoice
     * @return void
     *
     * Because during tax calculation, the sum of all Gw options is reset for the invoice tax amount and grand total,
     * we will add it at this point, at the end of the tax calculation.
     */
    private function addGwTaxes($invoice)
    {
        $baseTaxAmount = $invoice->getGwItemsBaseTaxAmount() + $invoice->getGwBaseTaxAmount() + $invoice->getGwCardBaseTaxAmount();
        $taxAmount = $invoice->getGwItemsTaxAmount() + $invoice->getGwTaxAmount() + $invoice->getGwCardTaxAmount();

        if ((float)$taxAmount > 0) {
            $invoice->setTaxAmount($invoice->getTaxAmount() + $taxAmount);
            $invoice->setGrandTotal($invoice->getGrandTotal() + $taxAmount);
        }
        if ((float)$baseTaxAmount > 0) {
            $invoice->setBaseTaxAmount($invoice->getBaseTaxAmount() + $baseTaxAmount);
            $invoice->setBaseGrandTotal($invoice->getBaseGrandTotal() + $baseTaxAmount);
        }
    }
}