<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-optimize
 * @version   2.1.13
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */



namespace Mirasvit\ImageLazyLoad\Processor;

use Mirasvit\ImageLazyLoad\Model\Config;
use Mirasvit\Optimize\Api\Processor\OutputProcessorInterface;

class LazyLoadProcessor implements OutputProcessorInterface
{
    CONST BG_CLASS_SUFFIX = '-mstlazy';

    private $config;

    private $imgCounter = 0;

    private $bgImageCounter = 0;

    private $backgroundImageClasses = [];

    public function __construct(
        Config $config
    ) {
        $this->config = $config;
    }

    /**
     * {@inheritdoc}
     */
    public function process($content)
    {
        if (!$this->config->isEnabled()) {
            return $content;
        }

        $content = preg_replace_callback(
            '/(<\s*img[^>]*)(src\s*=\s*["\'][^"\'<]+[\'"])([^>]{0,}>)/is',
            [$this, 'replaceImgNativeCallback'],
            $content
        );

        $content = preg_replace_callback(
            '/<[^>]*data-background-images\s*=\s*[\'"]{[^}]*}[\'"][^>]*>/is',
            [$this, 'replaceBackgroundImagesPagebuilderCallback'],
            $content
        );

        $bgClasses = $this->config->getBgLazyloadClasses();

        if (count($bgClasses)) {

            $content = preg_replace_callback(
                '/<[^>]*class\s*=\s*[\'"][^\'"]*(' . implode('|', $bgClasses) . ')[^\'"]*[\'"][^>]*>/is',
                [$this, 'replaceBackgroundImagesClassesCallback'],
                $content
            );
        }

        if (count($this->backgroundImageClasses)) {
            $content = $this->appendBackgroundLazyloadScript($content);
        }

        return $content;
    }

    private function replaceImgNativeCallback(array $match): string
    {
        $imageTag = preg_replace('/loading\s*=\s*[\'"]lazy[\'"]/', '', $match[0]);

        if (++$this->imgCounter <= $this->config->getSkipNumber()) {
            return $this->config->isAddFethpriorityHighToPreloadedImages()
                ? str_replace('<img', '<img fetchpriority="high" ', $imageTag)
                : $imageTag;
        }

        if ($this->config->isException($match[0])) {
            return $imageTag;
        }

        return str_replace('<img', '<img loading="lazy" ', $imageTag);
    }

    private function replaceBackgroundImagesClassesCallback(array $match): string
    {
        if (++$this->bgImageCounter <= $this->config->getSkipBgNumber()) {
            return $match[0];
        }

        if ($this->config->isException($match[0])) {
            return $match[0];
        }

        $newClass = $match[1] . self::BG_CLASS_SUFFIX;

        if (array_search($newClass, $this->backgroundImageClasses) === false) {
            $this->backgroundImageClasses[] = $newClass;
        }

        return str_replace($match[1], $newClass, $match[0]);
    }

    private function replaceBackgroundImagesPagebuilderCallback(array $match): string
    {
        if ($this->config->isException($match[0])) {
            return $match[0];
        }

        if (preg_match('/\s?class\s*=\s*[\'"]([^\'"]*)[\'"]/is', $match[0], $m)) {
            $classList = explode(' ', $m[1]);

            foreach ($classList as $idx => $class) {
                if (strpos($class, 'background-image-') !== false) {
                    $newClass = $class . self::BG_CLASS_SUFFIX;

                    if (array_search($newClass, $this->backgroundImageClasses) === false) {
                        $this->backgroundImageClasses[] = $newClass;
                    }

                    $classList[$idx] = $newClass;
                    break;
                }
            }

            return str_replace($m[1], implode(' ', $classList), $match[0]);
        } else {
            return $match[0];
        }
    }

    private function appendBackgroundLazyloadScript(string $content): string
    {
        $classes = implode(
            ', ',
            array_map(function ($cls) {return '.' . $cls;}, $this->backgroundImageClasses)
        );
        $suffix = self::BG_CLASS_SUFFIX;

        $script = "
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            var lazyBackgrounds = [].slice.call(document.querySelectorAll('{$classes}'));

            if ('IntersectionObserver' in window) {
                let lazyBackgroundObserver = new IntersectionObserver(function(entries, observer) {
                    entries.forEach(function(entry) {
                        if (entry.isIntersecting) {
                            let lazyloadClass = '';

                            entry.target.classList.forEach((exist) => {
                                if (exist.indexOf('-mstlazy') > 0) {
                                    lazyloadClass = exist;
                                    return;
                                }
                            });

                            if (lazyloadClass) {
                                entry.target.classList.remove(lazyloadClass);
                                entry.target.classList.add(lazyloadClass.replace('-mstlazy', ''))
                            }

                            entry.target.classList.add('lazyloaded');
                            lazyBackgroundObserver.unobserve(entry.target);
                        }
                    });
                });

                lazyBackgrounds.forEach(function(lazyBackground) {
                    lazyBackgroundObserver.observe(lazyBackground);
                });
            } else {
                /* fallback for old browsers */
                lazyBackgrounds.forEach(function(lazyBackground) {
                    let lazyloadClass = '';

                    lazyBackground.classList.forEach((exist) => {
                        if (exist.indexOf('{$suffix}') > 0) {
                            lazyloadClass = exist;
                            return;
                        }
                    });

                    if (lazyloadClass) {
                        lazyBackground.classList.remove(lazyloadClass);
                        lazyBackground.classList.add(lazyloadClass.replace('-mstlazy', ''))
                    }
                });
            }
        });
        </script>";

        return $content . $script;
    }
}
