<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-optimize
 * @version   2.1.13
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\OptimizeImage\Service;

use Magento\Framework\Exception\NotFoundException;
use Magento\Framework\Shell;
use Mirasvit\OptimizeImage\Api\Data\FileInterface;
use Mirasvit\OptimizeImage\Model\ConfigProvider;
use Mirasvit\OptimizeImage\Repository\FileRepository;

class WebpService
{
    private $shell;

    private $configProvider;

    private $validationService;

    private $fileService;

    private $fileRepository;

    private $index = 0;

    public function __construct(
        Shell $shell,
        ConfigProvider $configProvider,
        ValidationService $validationService,
        FileListSynchronizationService $fileService,
        FileRepository $fileRepository
    ) {
        $this->shell             = $shell;
        $this->configProvider    = $configProvider;
        $this->validationService = $validationService;
        $this->fileService       = $fileService;
        $this->fileRepository    = $fileRepository;
    }

    public function process(FileInterface $file): FileInterface
    {
        if (str_contains($file->getRelativePath(), '"')) {
            return $file;
        }

        if (!$this->validationService->canConvertWebp()) {
            return $file;
        }

        switch ($file->getFileExtension()) {
            case 'jpg':
            case 'jpeg':
            case 'png':
            case 'gif':
                $path = $this->generateWebp($file);
                $file->setWebpPath($path)->setWebpHash(sha1($path));

                break;
        }

        return $file;
    }

    /**
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    private function generateWebp(FileInterface $file): ?string
    {
        $path = $this->configProvider->getAbsolutePath($file->getRelativePath());

        if (!file_exists($path)) {
            throw new NotFoundException(__('The file was removed: %1', $path));
        }

        $optPath         = $this->configProvider->getOptPathFromRelativePath($file->getRelativePath());
        $optAbsolutePath = $this->configProvider->getAbsolutePath($optPath);
        $pathInfo        = pathinfo($optAbsolutePath);
        $fileExtension   = isset($pathInfo['extension']) ? strtolower($pathInfo['extension']) : '';

        $command = $fileExtension == 'gif'
            ? ConfigProvider::CMD_PROCESS_GIF2WEBP
            : ConfigProvider::CMD_PROCESS_WEBP;

        // file extension can be lowercase and uppercase (jpg, JPG)
        if (!strrpos($optPath, '.' . $fileExtension)) {
            $fileExtension = strtoupper($fileExtension);

            if (!strrpos($optPath, '.' . $fileExtension)) {
                return null;
            }
        }

        $this->fileService->ensureDir($optAbsolutePath);

        $configCompression = $this->configProvider->getCompressionLevel();
        $webpRelativePath  = $this->ensureWebpPath(substr($optPath, 0, strrpos($optPath, '.' . $fileExtension))) . ConfigProvider::WEBP_SUFFIX;
        $newPath           = $this->configProvider->getAbsolutePath($webpRelativePath);

        if (file_exists($newPath) && $file->getCompression() == $configCompression) {
            return $webpRelativePath;
        }

        try {
            $this->shell->execute(sprintf($command, $configCompression, $path, $newPath));
        } catch (\Exception $e) {
            if ($convertedPath = $this->normalize($path)) {
                $this->shell->execute(sprintf($command, $configCompression, $convertedPath, $newPath));
                unlink($convertedPath);
            }
        }

        return $webpRelativePath;
    }

    private function normalize(string $path): ?string
    {
        $convertedPath = $path . ConfigProvider::CONVERT_SUFFIX;

        try {
            $this->shell->execute(sprintf(ConfigProvider::CMD_CONVERT_RGB, $path, $convertedPath));

            return $convertedPath;
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getImageWebpUrl(string $imageUrl): string
    {
        if (strpos($imageUrl, '.webp') !== false) {
            return $imageUrl;
        }

        preg_match('/\?.*/is', $imageUrl, $query);

        $query = count($query) ? $query[0] : '';
        $value = str_replace($query, '', $imageUrl);

        $absolutePath = $this->configProvider->retrieveImageAbsPath($value);

        if (!$absolutePath) {
            return $imageUrl;
        }

        $image = $this->fileService->ensureFile($absolutePath);

        if ($image && $image->getWebpPath()) {
            $webpPath = $this->configProvider->getAbsolutePath($image->getWebpPath());

            if (!file_exists($webpPath)) {
                $image->setWebpPath(null)->setWebpHash(null);
                $this->fileRepository->save($image);

                return $imageUrl;
            }

            $webpUrl = $this->configProvider->getImageUrl($image->getWebpPath());

            $imageUrl = $webpUrl . $query;
        }

        return $imageUrl;
    }

    private function ensureWebpPath(string $webpRelativePath): string
    {
        $file = $this->fileRepository->getByWebpPath($webpRelativePath . ConfigProvider::WEBP_SUFFIX);

        if (!$file) {
            $this->index = 0;

            return $webpRelativePath;
        }

        $this->index++;

        return $this->ensureWebpPath($webpRelativePath . '_' . $this->index);
    }
}
