<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-search-ultimate
 * @version   2.3.13
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\Search\Service;

use Exception;
use Magento\Catalog\Model\Layer\Resolver;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Controller\Result\Raw;
use Magento\Framework\Controller\Result\RawFactory;
use Magento\Framework\Module\Manager as ModuleManager;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\View\Element\BlockInterface;
use Magento\Framework\View\Layout\Element;
use Magento\Framework\View\Result\Page;
use Mirasvit\Core\Service\CspService;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class AjaxResponseService
{
    private $resolver;

    private $resultRawFactory;

    private $request;

    private $moduleManager;

    private $serializer;

    public function __construct(
        Resolver      $resolver,
        RawFactory    $resultRawFactory,
        Http          $request,
        Json          $serializer,
        ModuleManager $moduleManager
    ) {
        $this->resolver         = $resolver;
        $this->resultRawFactory = $resultRawFactory;
        $this->request          = $request;
        $this->moduleManager    = $moduleManager;
        $this->serializer       = $serializer;
    }

    public function getAjaxResponse(Page $page): Raw
    {
        $data = $this->buildDataInstantMode($page);
        $data = $this->prepareAjaxData($data);

        return $this->createResponse($data);
    }

    protected function prepareAjaxData(array $data): array
    {
        $map = [
            '&amp;'         => '&',
            '?isAjax=1&'    => '?',
            '?isAjax=1'     => '',
            '&isAjax=1'     => '',
            '?isAjax=true&' => '?',
            '?isAjax=true'  => '',
            '&isAjax=true'  => '',
        ];

        foreach ($map as $search => $replace) {
            $data = str_replace($search, $replace, $data);
        }

        return $data;
    }

    protected function createResponse(array $data): Raw
    {
        return $this->resultRawFactory->create()
            ->setHeader('Content-type', 'text/plain')
            ->setContents($this->serializer->serialize($data));
    }

    private function buildDataInstantMode(Page $page): array
    {
        $layout        = $page->getLayout();
        $productsHtml  = $this->getProductsHtml($page);
        $productsCount = $this->getProductsCount();
        $leftNavHtml   = $this->getBlockHtml($page, 'catalog.leftnav', 'catalogsearch.leftnav');

        $categoryViewData = '';
        $children         = $layout->getChildNames('category.view.container');
        foreach ($children as $child) {
            $categoryViewData .= $layout->renderElement($child);
        }

        $categoryViewData = '<div class="category-view">' . $categoryViewData . '</div>';

        $data = [
            'products'         => $productsHtml,
            'products_count'   => $productsCount,
            'leftnav'          => $leftNavHtml,
            'categoryViewData' => $categoryViewData,
            'search_across'    => (string)__('Search across %1 products', $productsCount),
            'total_found'      => (string)__('found %1 products', $productsCount),
        ];

        if ($this->moduleManager->isEnabled('Mirasvit_QuickNavigation')) {
            $data['quickNavigation'] = $this->getBlockHtml($page, 'quick_navigation.filter_list');
        }

        try {
            $sidebarTag   = $layout->getElementProperty('div.sidebar.additional', Element::CONTAINER_OPT_HTML_TAG);
            $sidebarClass = $layout->getElementProperty('div.sidebar.additional', Element::CONTAINER_OPT_HTML_CLASS);

            if (method_exists($layout, 'renderNonCachedElement')) {
                $sidebarAdditional = $layout->renderNonCachedElement('div.sidebar.additional');
            } else {
                $sidebarAdditional = '';
            }

            $data['sidebarAdditional']         = $sidebarAdditional;
            $data['sidebarAdditionalSelector'] = $sidebarTag . '.' . str_replace(' ', '.', $sidebarClass);
        } catch (Exception $e) {
        }

        if ($this->moduleManager->isEnabled('Lof_AjaxScroll')) {
            $data['products'] .= $layout->createBlock('Lof\AjaxScroll\Block\Init')
                ->setTemplate('Lof_AjaxScroll::init.phtml')->toHtml();
            $data['products'] .= $layout->createBlock('Lof\AjaxScroll\Block\Init')
                ->setTemplate('Lof_AjaxScroll::scripts.phtml')->toHtml();
            $nonce = CspService::getNonce();
            $script = $nonce ? '<script nonce="' . $this->escapeHtml($nonce) . '">' : '<script>';
            $data['products'] .= $script . "window.ias.nextUrl = window.ias.getNextUrl();</script>";
        }

        if ($this->moduleManager->isEnabled('Mirasvit_Scroll')) {
            $data['products'] .= $layout->createBlock('Mirasvit\Scroll\Block\Scroll')
                ->setTemplate('Mirasvit_Scroll::scroll.phtml')->toHtml();
        }

        return $data;
    }

    private function getProductsHtml(Page $page): string
    {
        if (in_array($this->request->getFullActionName(), ['brand_brand_view', 'all_products_page_index_index'])) {
            $productsHtml = $this->getBlockHtml($page, 'category.products.list');
        } else {
            $productsHtml = $this->getBlockHtml($page, 'category.products', 'search.result');
        }

        return $productsHtml;
    }

    private function getProductsCount(): int
    {
        $productCollection = $this->resolver->get()->getProductCollection();

        return $productCollection ? $productCollection->getSize() : 0;
    }

    private function getBlockHtml(Page $page, string $blockName, string $fallbackBlockName = ''): string
    {
        $block = $this->getBlock($page, $blockName, $fallbackBlockName);

        return $block ? $block->toHtml() : '';
    }

    private function getBlock(Page $page, string $blockName, string $fallbackBlockName = ''): ?BlockInterface
    {
        $layout = $page->getLayout();
        $block  = $layout->getBlock($blockName);

        if (!$block && $fallbackBlockName) {
            $block = $layout->getBlock($fallbackBlockName);
        }

        return $block ? : null;
    }
}
