<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-search-ultimate
 * @version   2.3.13
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\SearchElastic\Plugin;

use Magento\Elasticsearch\Model\Adapter\Elasticsearch;
use Magento\Framework\App\ResourceConnection;
use Mirasvit\Search\Api\Data\ScoreRuleInterface;
use Mirasvit\Search\Model\ConfigProvider;

/**
 * @SuppressWarnings(PHPMD)
 * @see \Magento\Elasticsearch\Model\Adapter\Elasticsearch::addDocs()
 */
class PutScoreBoostBeforeAddDocsPlugin
{
    const EMPTY_SUM_VALUE = 0;

    const EMPTY_MULTIPLY_VALUE = 1;

    private $resource;

    public function __construct(
        ResourceConnection $resource
    ) {
        $this->resource = $resource;
    }

    public function beforeAddDocs(Elasticsearch $subject, array $docs, int $storeId, string $mappedIndexerId)
    {
        if ($mappedIndexerId == 'mst_misspell_index') {
            return [$docs, $storeId, $mappedIndexerId];
        }

        $productIds = array_keys($docs);

        $connection = $this->resource->getConnection();
        $select     = $connection->select()->from(['index' => $this->getIndexTable()], ['product_id', 'score_factor'])
            ->where('index.store_id IN (?)', [0, $storeId])
            ->where('index.product_id IN (?)', $productIds);

        $rows = $connection->fetchAll($select);

        $scoreFactors = [];
        array_map(function ($row) use (&$scoreFactors) {
            $scoreFactors[$row['product_id']] = $row['score_factor'];
        }, $rows);

        foreach ($docs as $productId => $doc) {
            $docs[$productId][ConfigProvider::SUM_ATTRIBUTE]      = self::EMPTY_SUM_VALUE;
            $docs[$productId][ConfigProvider::MULTIPLY_ATTRIBUTE] = self::EMPTY_MULTIPLY_VALUE;

            if (isset($scoreFactors[$productId])) {
                if (strripos($scoreFactors[$productId], '*') !== false) {
                    $docs[$productId][ConfigProvider::MULTIPLY_ATTRIBUTE] = (int)str_replace('*', '', $scoreFactors[$productId]);
                } elseif (strripos($scoreFactors[$productId], '/') !== false) {
                    $docs[$productId][ConfigProvider::MULTIPLY_ATTRIBUTE] = 1.0 / (int)str_replace('/', '', $scoreFactors[$productId]);
                } else {
                    $docs[$productId][ConfigProvider::SUM_ATTRIBUTE] = (int)str_replace('+', '', $scoreFactors[$productId]);
                }
            }
        }

        return [$docs, $storeId, $mappedIndexerId];
    }

    private function getIndexTable()
    {
        return $this->resource->getTableName(ScoreRuleInterface::INDEX_TABLE_NAME);
    }
}
