<?php
/**
 * Copyright (c) 2020. Volodymyr Hryvinskyi.  All rights reserved.
 * @author: <mailto:volodymyr@hryvinskyi.com>
 * @github: <https://github.com/hryvinskyi>
 */

namespace MageCloud\SeoRatingSnippet\ViewModel;

use MageCloud\SeoRatingSnippet\ViewModel\Category;
use Magento\Catalog\Model\Category as CategoryModel;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ResourceModel\Product\Collection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Registry;
use Magento\Store\Model\StoreManagerInterface;

class Offers extends Category
{
    /**
     * @var StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var CollectionFactory
     */
    private $productCollectionFactory;
    /**
     * @var \Magento\Catalog\Helper\Data
     */
    private $taxHelper;

    /**
     * @var \Magento\Catalog\Model\Product
     */
    private $product;

    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * Offers constructor.
     *
     * @param Registry $registry
     * @param RequestInterface $request
     * @param StoreManagerInterface $storeManager
     * @param CollectionFactory $productCollectionFactory
     */
    public function __construct(
        Registry $registry,
        RequestInterface $request,
        StoreManagerInterface $storeManager,
        CollectionFactory $productCollectionFactory,
        \Magento\Catalog\Helper\Data $taxHelper,
        \Magento\Catalog\Model\Product $product,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
    ) {
        parent::__construct($registry, $request);

        $this->storeManager = $storeManager;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->taxHelper = $taxHelper;
        $this->product = $product;
        $this->productRepository = $productRepository;
    }

    /**
     * @return string
     */
    public function getPriceCurrency(): string
    {
        $result = '';

        if ($this->isCategoryPage() !== false) {
            $result = $this->storeManager->getStore()->getCurrentCurrency()->getCode();
        }

        return $result;
    }

    /**
     * @return float
     */
    public function getLowPrice(): float
    {
        $result = .0;

        if ($this->isCategoryPage() === false) {
            return $result;
        }

        $category = $this->getCategory();

        $collection = $this->getProductCollection($category);
        $collection = $collection->setOrder('price', 'asc')->getFirstItem();

        $product_id = $this->productRepository->getById($collection->getId());
        //$product_id = $this->product->load($collection->getId());
        $taxx_simpl = $this->taxHelper->getTaxPrice($product_id, $collection->getFinalPrice(), true);
        $taxx_conf = $this->taxHelper->getTaxPrice($product_id, $collection->getMinPrice(), true);


        if ($collection->getId()) {
            $data = $collection->getData();
            $result = $taxx_simpl?$taxx_simpl:$data['final_price'];

            if ($data['type_id'] === 'configurable') {
                $result = $taxx_conf?$taxx_conf:$data['min_price'];
            }
        }

        return (float)$result;
    }

    /**
     * @return float
     */
    public function getHighPrice(): float
    {
        $result = .0;

        if ($this->isCategoryPage() === false) {
            return $result;
        }

        $category = $this->getCategory();

        $collection = $this->getProductCollection($category);
        $collection = $collection->setOrder('price', 'desc')->getFirstItem();

        $product_id = $this->productRepository->getById($collection->getId());
        $taxx_simpl = $this->taxHelper->getTaxPrice($product_id, $collection->getFinalPrice(), true);
        $taxx_conf = $this->taxHelper->getTaxPrice($product_id, $collection->getMinPrice(), true);

        if ($collection->getId()) {
            $data = $collection->getData();
            $result = $taxx_simpl?$taxx_simpl:$data['final_price'];

            if ($data['type_id'] === 'configurable') {
                $result = $taxx_conf?$taxx_conf:$data['max_price'];
            }
        }

        return $result;
    }

    /**
     * @param CategoryModel $category
     *
     * @return mixed
     */
    public function getProductCollection(CategoryModel $category): Collection
    {
        $collection = $this->productCollectionFactory->create();
        $collection
            ->addFinalPrice()
            ->addCategoryFilter($category)
            ->addAttributeToFilter('visibility', Visibility::VISIBILITY_BOTH)
            ->addAttributeToFilter('status', Status::STATUS_ENABLED);
        return $collection;
    }

    /**
     * @return Collection
     */
    public function getProductCollectionCount(): Collection
    {
        $collection = $this->productCollectionFactory->create();
        $collection
            ->addFinalPrice()
            ->addCategoryFilter($this->getCategory())
            ->addAttributeToFilter('visibility', Visibility::VISIBILITY_BOTH)
            ->addAttributeToFilter('status', Status::STATUS_ENABLED);
        return $collection;
    }
}
