<?php

namespace StripeIntegration\Payments\Observer;

use Magento\Framework\Event\ObserverInterface;

class AddInitialFeeTaxObserver implements ObserverInterface
{
    public $helper = null;
    public $taxHelper = null;
    public $subscriptionsHelper = null;

    private $config;
    private $paymentsHelperFactory;
    private $subscriptionsHelperFactory;
    private $subscriptionProductFactory;

    public function __construct(
        \StripeIntegration\Payments\Helper\GenericFactory $paymentsHelper,
        \StripeIntegration\Payments\Helper\SubscriptionsFactory $subscriptionsHelperFactory,
        \StripeIntegration\Payments\Model\SubscriptionProductFactory $subscriptionProductFactory,
        \StripeIntegration\Payments\Model\Config $config
    )
    {
        $this->paymentsHelperFactory = $paymentsHelper;
        $this->subscriptionsHelperFactory = $subscriptionsHelperFactory;
        $this->config = $config;
        $this->subscriptionProductFactory = $subscriptionProductFactory;
    }

    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        if (!$this->config->isSubscriptionsEnabled())
            return $this;

        $total = $observer->getData('total');
        $quote = $observer->getData('quote');

        if ($total && $total->getInitialFeeAmount() > 0)
            $this->applyInitialFeeTax($quote, $total);

        return $this;
    }

    public function applyInitialFeeTax($quote, $total)
    {
        if ($this->config->priceIncludesTax())
            return;

        $baseExtraTax = 0;
        $extraTax = 0;

        if (!$this->helper)
            $this->helper = $this->paymentsHelperFactory->create();

        if (!$this->subscriptionsHelper)
            $this->subscriptionsHelper = $this->subscriptionsHelperFactory->create();

        foreach ($quote->getAllItems() as $item)
        {
            $subscriptionProductModel = $this->subscriptionProductFactory->create()->fromQuoteItem($item);
            if (!$subscriptionProductModel->isSubscriptionProduct())
                continue;

            if (!$quote->getQuoteCurrencyCode() || !$quote->getBaseCurrencyCode())
            {
                $quote->beforeSave(); // Sets the currencies
            }

            $initialFeeDetails = $this->subscriptionsHelper->getInitialFeeDetails($subscriptionProductModel->getProduct(), $quote, $item);

            $baseExtraTax += $initialFeeDetails['base_tax'];
            $extraTax += $initialFeeDetails['tax'];
        }

        $total->addTotalAmount('tax', $extraTax);
        $total->addBaseTotalAmount('tax', $baseExtraTax);
        $total->setGrandTotal(round(floatval($total->getGrandTotal()) + floatval($extraTax), 4));
        $total->setBaseGrandTotal(round(floatval($total->getBaseGrandTotal()) + floatval($baseExtraTax), 2));
    }
}
