<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-optimize
 * @version   2.1.13
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\OptimizeImage\Cron;


use Mirasvit\OptimizeImage\Model\ConfigProvider;
use Mirasvit\OptimizeImage\Repository\FileRepository;
use Mirasvit\OptimizeImage\Service\FileListBatchService;
use Mirasvit\OptimizeImage\Service\FileListSynchronizationService;
use Mirasvit\OptimizeImage\Service\OptimizeService;
use Mirasvit\OptimizeImage\Service\WebpService;

class OptimizeCron
{
    private $fileListSynchronizationService;

    private $fileRepository;

    private $fileListBatchService;

    private $optimizeService;

    private $webpService;

    private $configProvider;

    public function __construct(
        FileListSynchronizationService $fileListSynchronizationService,
        FileRepository $fileRepository,
        FileListBatchService $fileListBatchService,
        OptimizeService $optimizeService,
        WebpService $webpService,
        ConfigProvider $configProvider
    ) {
        $this->fileListSynchronizationService = $fileListSynchronizationService;
        $this->fileRepository                 = $fileRepository;
        $this->fileListBatchService           = $fileListBatchService;
        $this->optimizeService                = $optimizeService;
        $this->webpService                    = $webpService;
        $this->configProvider                 = $configProvider;
    }

    /**
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    public function execute()
    {
        if ($this->configProvider->isFilesystemStrategy()) {
            $this->fileListSynchronizationService->synchronize(10000);
        }

        while ($batch = $this->fileListBatchService->getBatch()) {
            foreach ($batch as $file) {
                try {
                    $path = $file->getRelativePath();

                    $this->optimizeService->optimize($file);

                    if ($optPath = $file->getOptimizedPath()) {
                        if (!file_exists($this->configProvider->getAbsolutePath($optPath))) {
                            usleep(100);
                        }

                        $path = $optPath;
                    }

                    if ($this->configProvider->isWebpEnabled()) {
                        $this->webpService->process($file);

                        if ($webpPath = $file->getWebpPath()) {
                            if (!file_exists($this->configProvider->getAbsolutePath($webpPath))) {
                                usleep(100);
                            }

                            $path = $webpPath;
                        }
                    }

                    if ($file->getRelativePath() !== $path) {
                        $file->setActualSize(filesize($this->configProvider->getAbsolutePath($path)))
                            ->setCompression($this->configProvider->getCompressionLevel());
                    }

                    $file->setProcessedAt(date('Y-m-d H:i:s'));
                    $this->fileRepository->save($file);
                } catch (\Exception $e) {
                    $this->fileRepository->delete($file);
                }
            }
        }
    }
}
