<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-search-ultimate
 * @version   2.3.13
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */



namespace Mirasvit\Search\Model\ScoreRule\Indexer;

use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Ddl\Table;
use Magento\Framework\Indexer\ActionInterface as IndexerActionInterface;
use Mirasvit\Search\Api\Data\ScoreRuleInterface;
use Mirasvit\Search\Repository\ScoreRuleRepository;

class ScoreRuleIndexer implements IndexerActionInterface
{
    const INDEXER_ID = 'mirasvit_search_score_rule_product';

    const RULE_ID      = ScoreRuleInterface::ID;
    const STORE_ID     = 'store_id';
    const PRODUCT_ID   = 'product_id';
    const SCORE_FACTOR = ScoreRuleInterface::SCORE_FACTOR;

    private $resource;

    private $scoreRuleRepository;

    public function __construct(
        ResourceConnection  $resource,
        ScoreRuleRepository $scoreRuleRepository
    ) {
        $this->resource            = $resource;
        $this->scoreRuleRepository = $scoreRuleRepository;
    }

    public function executeFull()
    {
        $connection = $this->resource->getConnection();
        $connection->truncateTable($this->getIndexTable());
        $connection->query('ALTER TABLE ' . $this->getIndexTable() . ' AUTO_INCREMENT = 1');

        foreach ($this->scoreRuleRepository->getCollection() as $scoreRule) {
            $this->execute($scoreRule, []);
        }

        $this->executeZeroRule([]);
    }

    public function executeList(array $ids)
    {
        foreach ($this->scoreRuleRepository->getCollection() as $scoreRule) {
            $this->execute($scoreRule, $ids);
        }

        $this->executeZeroRule($ids);
    }

    public function executeRow($id)
    {
        foreach ($this->scoreRuleRepository->getCollection() as $scoreRule) {
            $this->execute($scoreRule, [$id]);
        }

        $this->executeZeroRule([$id]);
    }

    public function execute(ScoreRuleInterface $scoreRule, array $productIds)
    {
        if (!$scoreRule->isActive()) {
            return;
        }

        if ($scoreRule->getActiveFrom() && strtotime($scoreRule->getActiveFrom()) > time()) {
            return;
        }

        if ($scoreRule->getActiveTo() && strtotime($scoreRule->getActiveTo()) < time()) {
            return;
        }

        $connection = $this->resource->getConnection();

        $this->ensureIndexTable();

        // Real Score Rules
        foreach ($scoreRule->getStoreIds() as $storeId) {
            $storeId = intval($storeId);

            $deleteWhere = [
                self::STORE_ID . ' = ' . $storeId,
                self::RULE_ID . ' = ' . $scoreRule->getId(),
            ];
            if ($productIds) {
                $deleteWhere[] = self::PRODUCT_ID . ' IN(' . implode(',', $productIds) . ')';
            }

            $connection->delete($this->getIndexTable(), $deleteWhere);

            $ids = $scoreRule->getRule()->getMatchingProductIds($productIds);
            $ids = array_unique($ids);

            $scoreFactors = $scoreRule->getRule()->getScoreFactors($scoreRule, $ids, $storeId);

            $idx = 0;
            do {
                $rows = [];

                for (; $idx < count($ids); $idx++) {
                    $row = [
                        self::RULE_ID      => $scoreRule->getId(),
                        self::STORE_ID     => $storeId,
                        self::PRODUCT_ID   => $ids[$idx],
                        self::SCORE_FACTOR => $scoreFactors[$ids[$idx]],
                    ];

                    $rows[] = $row;

                    if (count($rows) > 1000) {
                        break;
                    }
                }

                if (count($rows)) {
                    $connection->insertMultiple($this->getIndexTable(), $rows);
                }
            } while (count($rows));
        }
    }

    private function ensureIndexTable()
    {
        $tableName = $this->getIndexTable();

        $connection = $this->resource->getConnection();

        if ($connection->isTableExists($tableName)) {
            return $this;
        }

        $table = $connection->newTable($tableName);

        $table->addColumn(self::RULE_ID, Table::TYPE_INTEGER);
        $table->addColumn(self::STORE_ID, Table::TYPE_INTEGER);
        $table->addColumn(self::PRODUCT_ID, Table::TYPE_INTEGER);
        $table->addColumn(self::SCORE_FACTOR, Table::TYPE_TEXT);

        $connection->createTable($table);

        return $this;
    }

    private function getIndexTable(): string
    {
        return $this->resource->getTableName(ScoreRuleInterface::INDEX_TABLE_NAME);
    }

    private function executeZeroRule(array $productIds)
    {
        $connection = $this->resource->getConnection();

        $this->ensureIndexTable();

        $deleteWhere = [
            self::STORE_ID . ' = 0',
            self::RULE_ID . ' = 0',
        ];
        if ($productIds) {
            $deleteWhere[] = self::PRODUCT_ID . ' IN(' . implode(',', $productIds) . ')';
        }

        $connection->delete($this->getIndexTable(), $deleteWhere);

        // Product Search Weight
        $select = $connection->select()->from(
            [$this->resource->getTableName('catalog_product_entity')],
            ['entity_id', 'mst_search_weight']
        )->where('(mst_search_weight > 0 or mst_search_weight < 0)');
        if ($productIds) {
            $select->where('entity_id IN(' . implode(',', $productIds) . ')');
        }

        $data = $connection->fetchAll($select);
        $idx  = 0;

        do {
            $rows = [];
            for (; $idx < count($data); $idx++) {
                $id     = $data[$idx]['entity_id'];
                $factor = $data[$idx]['mst_search_weight'];
                $row    = [
                    self::RULE_ID      => 0,
                    self::STORE_ID     => 0,
                    self::PRODUCT_ID   => $id,
                    self::SCORE_FACTOR => $factor > 0 ? '+' . $factor : $factor,
                ];

                $rows[] = $row;

                if (count($rows) > 1000) {
                    break;
                }
            }

            if (count($rows)) {
                $connection->insertMultiple($this->getIndexTable(), $rows);
            }
        } while (count($rows));
    }
}
