<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-search-ultimate
 * @version   2.3.13
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\Search\Service;

use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\ResourceConnection;
use Magento\Store\Model\StoreManagerInterface;
use Mirasvit\Search\Service\ContentService;

class MapperService
{
    const IN_STOCK = 2;

    const OUT_OF_STOCK = 1;

    const UNSET_STOCK = 0;

    private $resource;

    private $storeManager;

    private $contentService;

    public function __construct(
        ResourceConnection    $resource,
        StoreManagerInterface $storeManager,
        ContentService        $contentService
    ) {
        $this->resource       = $resource;
        $this->storeManager   = $storeManager;
        $this->contentService = $contentService;
    }

    public function getBaseUrl(int $storeId): string
    {
        return $this->storeManager->getStore($storeId)->getBaseUrl();
    }

    public function getAdminPath(): string
    {
        $url = ObjectManager::getInstance()->get(\Magento\Backend\Helper\Data::class)
            ->getHomePageUrl();

        $components = parse_url($url);
        $components = explode('/', trim($components['path'], '/'));

        return array_shift($components);
    }

    public function clearString(int $storeId, string $string, bool $processHtmlContent): string
    {
        if ($processHtmlContent) {
            $string = $this->contentService->processHtmlContent($storeId, $string);
        }

        $string = (string)preg_replace("/<style\\b[^>]*>(.*?)<\\/style>/s", "", html_entity_decode($string));

        return trim(strip_tags($string));
    }

    public function correctUrl(int $storeId, string $url): string
    {
        $baseUrl = $this->getBaseUrl($storeId);

        if (strripos($url, $baseUrl) === false || strripos($url, $this->getAdminPath()) !== false) {
            $url = str_replace('/' . $this->getAdminPath() . '/', '/', $url);
            $url = preg_replace('~\/key\/.*~', '', $url);
        }

        if (strripos($url, $baseUrl) === false) {
            $baseChunks = explode('/', $baseUrl);
            $urlChunks  = explode('/', $url);
            foreach ($baseChunks as $idx => $chunk) {
                if (isset($urlChunks[$idx])) {
                    $urlChunks[$idx] = $chunk;
                }
            }

            $url = implode('/', $urlChunks);
        }

        return $url;
    }

    public function addStockStatusInfo(int $storeId, array $productIds): array
    {
        $map = [];

        foreach ($productIds as $productId) {
            $map[$productId] = self::UNSET_STOCK;
        }

        $data = $this->resource->getConnection()->fetchPairs(
            $this->resource->getConnection()->select()
                ->from(
                    ['e' => $this->resource->getTableName('catalog_product_entity')],
                    ['entity_id']
                )->joinInner(
                    ['stock' => $this->resource->getTableName('cataloginventory_stock_status')],
                    'stock.product_id = e.entity_id',
                    ['stock_status']
                )
                ->where('e.entity_id IN (?)', $productIds)
                ->group('e.entity_id')
        );

        foreach ($data as $productId => $stockStatus) {
            $map[$productId] = $stockStatus == 1 ? self::IN_STOCK : self::OUT_OF_STOCK;
        }

        if ($this->resource->getConnection()->isTableExists($this->resource->getTableName('inventory_source_item'))) {
            $data = $this->resource->getConnection()->fetchAll(
                $this->resource->getConnection()->select()
                    ->from(
                        ['s' => $this->resource->getTableName('store')],
                        ['website_id', 'store_id']
                    )->joinInner(
                        ['sw' => $this->resource->getTableName('store_website')],
                        'sw.website_id = s.website_id',
                        []
                    )->joinInner(
                        ['issc' => $this->resource->getTableName('inventory_stock_sales_channel')],
                        'issc.code = sw.code',
                        null
                    )->joinInner(
                        ['issl' => $this->resource->getTableName('inventory_source_stock_link')],
                        'issl.stock_id = issc.stock_id',
                        ['stock_id']
                    )
                    ->where('s.store_id = ?', $storeId)
                    ->group('s.store_id')
            );

            $result = [];

            foreach ($data as $row) {

                $stockSelect = $this->resource->getConnection()->select();
                $stockId     = $row['stock_id'];

                if ($this->resource->getConnection()->isTableExists($this->resource->getTableName("inventory_stock_$stockId"))){
                    if($this->resource->getConnection()->tableColumnExists($this->resource->getTableName("inventory_stock_$stockId"), 'product_id')) {
                        $stockSelect->from(
                            ['stock' => $this->resource->getTableName("inventory_stock_$stockId")],
                            ['value' => 'is_salable', 'entity_id' => 'product_id']
                        );

                        if ($productIds) {
                            $stockSelect->where('stock.product_id IN (?)', $productIds);
                        }
                    } else {
                        $stockSelect->from(
                            ['stock' => $this->resource->getTableName("inventory_stock_$stockId")],
                            ['value' => 'is_salable']
                        )->joinInner(
                            ['e' => $this->resource->getTableName('catalog_product_entity')],
                            'e.sku = stock.sku',
                            ['entity_id']
                        );

                        if ($productIds) {
                            $stockSelect->where('e.entity_id IN (?)', $productIds);
                        }
                    }
                } else {
                    $stockSelect->from(
                        ['stock' => $this->resource->getTableName("cataloginventory_stock_item")],
                        ['value' => 'is_in_stock']
                    )->joinInner(
                        ['e' => $this->resource->getTableName('catalog_product_entity')],
                        'e.entity_id = stock.product_id',
                        ['entity_id']
                    );

                    if ($productIds) {
                        $stockSelect->where('e.entity_id IN (?)', $productIds);
                    }
                }

                foreach ($this->resource->getConnection()->query($stockSelect)->fetchAll() as $stock) {
                    $stock    = array_merge($stock, $row);
                    $result[] = $stock;
                };
            }

            foreach ($result as $item) {
                $map[(int)$item['entity_id']] = (int)$item['value'] == 1 ? self::IN_STOCK : self::OUT_OF_STOCK;
            }
        }

        return $map;
    }

}
