<?php

namespace StripeIntegration\Tax\Model\StripeTransactionReversal\Request;

use StripeIntegration\Tax\Helper\GiftOptions;
use StripeIntegration\Tax\Helper\Logger;
use StripeIntegration\Tax\Model\Config;
use StripeIntegration\Tax\Model\StripeTransactionReversal\Request\LineItems\LineItem;
use StripeIntegration\Tax\Helper\Order;
use Stripe\Collection;

class LineItems
{
    private $data = [];
    private $lineItem;
    private $config;
    private $logger;
    private $giftOptionsHelper;
    private $includeOrderGW;
    private $includeOrderPrintedCard;
    private $orderHelper;
    private $lineItemsData;
    private $lineItemsHelper;

    public function __construct(
        LineItem $lineItem,
        Config $config,
        Logger $logger,
        GiftOptions $giftOptionsHelper,
        Order $orderHelper,
        \StripeIntegration\Tax\Helper\LineItems $lineItemsHelper
    )
    {
        $this->lineItem = $lineItem;
        $this->config = $config;
        $this->logger = $logger;
        $this->giftOptionsHelper = $giftOptionsHelper;
        $this->includeOrderGW = false;
        $this->includeOrderPrintedCard = false;
        $this->orderHelper = $orderHelper;
        $this->lineItemsHelper = $lineItemsHelper;
    }

    private function clearData()
    {
        $this->data = [];
    }
    public function formOnlineData($creditMemo)
    {
        $this->clearData();
        $transactionId = $creditMemo->getInvoice()->getStripeTaxTransactionId();
        try {
            /** @var Collection $transactionLineItems */
            $transactionLineItems = $this->config->getStripeClient()->tax->transactions->allLineItems($transactionId);
        } catch (\Exception $e) {
            $this->logger->logError(sprintf('Unable to retrieve line items for transaction %s: ', $transactionId) . $e->getMessage());
        }

        if (!empty($transactionLineItems->data) && $transactionLineItems->getLastResponse()->code === 200) {
            foreach ($creditMemo->getAllItems() as $item) {
                $orderItem = $item->getOrderItem();
                if ($orderItem->isDummy() || $item->getQty() <= 0) {
                    // in case the order item is a bundle dynamic product, add the GW options to the request before skipping
                    if ($orderItem->getHasChildren() &&
                        $orderItem->isChildrenCalculated() &&
                        $this->giftOptionsHelper->itemHasGiftOptions($orderItem)
                    ) {
                        $this->lineItem->formItemGwData($item, $orderItem, $creditMemo, $transactionLineItems->data);
                        $this->data[] = $this->lineItem->toArray();
                    }
                    continue;
                }
                $this->lineItem->formData($item, $creditMemo, $transactionLineItems->data);
                $this->data[] = $this->lineItem->toArray();
                if ($this->giftOptionsHelper->itemHasGiftOptions($orderItem)) {
                    $this->lineItem->formItemGwData($item, $orderItem, $creditMemo, $transactionLineItems->data);
                    $this->data[] = $this->lineItem->toArray();
                }
            }
            if ($this->includeOrderGW) {
                $this->lineItem->formOrderGwData($creditMemo, $transactionLineItems->data);
                $this->data[] = $this->lineItem->toArray();
            }
            if ($this->includeOrderPrintedCard) {
                $this->lineItem->formOrderPrintedCardData($creditMemo, $transactionLineItems->data);
                $this->data[] = $this->lineItem->toArray();
            }
        }
    }

    public function formOfflineData($creditMemo, $invoice, $transaction)
    {
        $this->clearData();

        // Initialize the linen items details before checking for items and totals
        // Added before the check because there can be credit memos containing only shipping and this will
        // make it so line items details are set on the reversal details on order
        $this->initLineItemsData($creditMemo, $transaction);

        // Start going through the items only if the transaction has line items or the credit memo hasn't met
        // the amount necessary to be reverted
        if ($transaction->hasLineItems() && $creditMemo->getGrandTotal() > $creditMemo->getAmountToRevert()) {
            foreach ($creditMemo->getAllItems() as $item) {
                $orderItem  = $item->getOrderItem();
                if (!$item->getAmountReverted()) {
                    $item->setAmountReverted(0);
                }
                if (!$item->getAmountTaxReverted()) {
                    $item->setAmountTaxReverted(0);
                }
                $reference = $this->lineItemsHelper->getReferenceForInvoiceTax($item, $creditMemo->getOrder());
                // If the credit memo item is not found in the transaction items or
                // if there is no more amount available for the item to revert from this transaction or
                // if the credit memo item is already reverted in previous transactions, go to the next credit memo item
                if ($orderItem->isDummy() ||
                    $item->getQty() <= 0 ||
                    !isset($this->lineItemsData[$reference]) ||
                    ($this->lineItemsData[$reference]['remaining_amount'] <= 0 && $this->lineItemsData[$reference]['remaining_amount_tax'] <= 0) ||
                    ($this->lineItemsHelper->getAmount($item, $creditMemo->getOrderCurrencyCode()) <= $item->getAmountReverted() &&
                        $this->lineItemsHelper->getStripeFormattedAmount($item->getTaxAmount(), $creditMemo->getOrderCurrencyCode()) <= $item->getAmountTaxReverted())
                ) {
                    // in case the order item is a bundle dynamic product, add the GW options to the request before skipping
                    if ($orderItem->getHasChildren() &&
                        $orderItem->isChildrenCalculated() &&
                        $this->giftOptionsHelper->itemHasGiftOptions($orderItem)
                    ) {
                        $this->lineItem->formOfflineItemGwData($item, $orderItem, $creditMemo, $transaction->getLineItems());
                        $this->processGiftOptionsLineItem($transaction, false);
                    }
                    continue;
                }
                $this->lineItem->formOfflineData($item, $creditMemo, $this->lineItemsData[$reference]);

                $this->updateRemainingAmounts($reference, $this->lineItem->getAmount(), $this->lineItem->getAmountTax());

                $this->data[] = $this->lineItem->toArray();
                $transaction->addItemProcessed();

                if ($this->giftOptionsHelper->itemHasGiftOptions($orderItem)) {
                    $this->lineItem->formOfflineItemGwData($item, $orderItem, $creditMemo, $transaction->getLineItems());
                    $this->processGiftOptionsLineItem($transaction);
                }

                // If all the items in the transaction are processed or the total to be refunded is reached, stop the
                // loop through the credit memo items
                if ($transaction->isProcessed() || ($creditMemo->getGrandTotal() == $creditMemo->getAmountToRevert())) {
                    break;
                }
            }
            if ($this->includeOrderGW && $this->giftOptionsHelper->invoiceHasGw($invoice)) {
                $this->lineItem->formOfflineOrderGwData($creditMemo, $transaction->getLineItems());
                $this->processGiftOptionsLineItem($transaction);
            }
            if ($this->includeOrderPrintedCard && $this->giftOptionsHelper->invoiceHasPrintedCard($invoice)) {
                $this->lineItem->formOfflineOrderPrintedCardData($creditMemo, $transaction->getLineItems());
                $this->processGiftOptionsLineItem($transaction);
            }
        }

        return !$this->hasRemainingAmount();
    }

    public function hasRemainingAmount()
    {
        foreach ($this->lineItemsData as $lineItem) {
            if ($lineItem['remaining_amount'] > 0 && $lineItem['remaining_amount_tax'] > 0) {
                return true;
            }
        }

        return false;
    }

    public function toArray()
    {
        return $this->data;
    }

    public function canIncludeInRequest()
    {
        return !empty($this->data);
    }

    public function setIncludeOrderGW($includeOrderGW)
    {
        $this->includeOrderGW = $includeOrderGW;

        return $this;
    }

    public function setIncludeOrderPrintedCard($includeOrderPrintedCard)
    {
        $this->includeOrderPrintedCard = $includeOrderPrintedCard;

        return $this;
    }

    private function initLineItemsData($creditMemo, $transaction)
    {
        $lineItemsData = $this->orderHelper->getReversalLineItemsData($creditMemo->getOrder(), $transaction->getId());
        if (!$lineItemsData) {
            $lineItemsData = $transaction->formLineItemsData();
        }

        $this->lineItemsData = $lineItemsData;
    }

    public function getLineItemsData()
    {
        return $this->lineItemsData;
    }

    private function processGiftOptionsLineItem($transaction, $addItemAsProcessed = true)
    {
        $this->updateRemainingAmounts($this->lineItem->getReference(), $this->lineItem->getAmount(), $this->lineItem->getAmountTax());
        $this->data[] = $this->lineItem->toArray();
        if ($addItemAsProcessed) {
            $transaction->addItemProcessed();
        }
    }

    private function updateRemainingAmounts($reference, $amountReverted, $taxAmountReverted)
    {
        $this->lineItemsData[$reference]['remaining_amount'] = $this->lineItemsData[$reference]['remaining_amount'] + $amountReverted;
        $this->lineItemsData[$reference]['remaining_amount_tax'] = $this->lineItemsData[$reference]['remaining_amount_tax'] + $taxAmountReverted;
    }
}
