<?php

namespace StripeIntegration\Tax\Plugin\Sales\Model\Order\Invoice\Total;

use Magento\Directory\Model\CurrencyFactory;
use StripeIntegration\Tax\Helper\GiftOptions;
use StripeIntegration\Tax\Helper\LineItems;
use StripeIntegration\Tax\Helper\TaxCalculator;
use StripeIntegration\Tax\Model\StripeTax;
use \Magento\Sales\Model\Order\Invoice\Total\Tax as InvoiceTax;
use \Magento\Sales\Model\Order\Invoice;
use StripeIntegration\Tax\Model\Order\Transaction\Item;

class Tax
{
    private $stripeTax;
    private $transactionItem;
    private $taxCalculationHelper;
    private $currencyFactory;
    private $lineItemsHelper;
    private $invoiceHelper;
    private $giftOptionsHelper;

    public function __construct(
        StripeTax $stripeTax,
        Item $transactionItem,
        TaxCalculator $taxCalculationHelper,
        CurrencyFactory $currencyFactory,
        LineItems $lineItemsHelper,
        \StripeIntegration\Tax\Helper\Invoice $invoiceHelper,
        GiftOptions $giftOptionsHelper
    )
    {
        $this->stripeTax = $stripeTax;
        $this->transactionItem = $transactionItem;
        $this->taxCalculationHelper = $taxCalculationHelper;
        $this->currencyFactory = $currencyFactory;
        $this->lineItemsHelper = $lineItemsHelper;
        $this->invoiceHelper = $invoiceHelper;
        $this->giftOptionsHelper = $giftOptionsHelper;
    }

    /**
     * @param InvoiceTax $subject
     * @param Invoice $invoice
     * @return Invoice[]
     *
     * Before the taxes are calculated, we run the Stripe tax calculations, so that we have the initial prices of the
     * products in the order. After the values are calculated we save them either on the item or invoice,
     * depending on the place where we are.
     */
    public function beforeCollect(InvoiceTax $subject, Invoice $invoice)
    {
        if ($this->stripeTax->isEnabled()) {
            $this->stripeTax->calculateForInvoiceTax($invoice->getOrder(), $invoice);
            if ($this->stripeTax->hasValidResponse()) {
                $this->setStripeCalculatedValues($invoice);
            }
        }

        return [$invoice];
    }

    /**
     * @param InvoiceTax $subject
     * @param $result
     * @param Invoice $invoice
     * @return mixed
     *
     * After the core Magento tax collection is run, we will add the details which will be set by the Stripe Tax
     * calculation. We will set these values in separate 'initial' properties of the invoice and invoice item objects.
     * Setting these values will ensure that if other calculations will occur, we will be able to subtract these
     * values from the invoice and invoice items when we will update the invoice with the Stripe calculated values.
     */
    public function afterCollect(
        InvoiceTax $subject,
                   $result,
        Invoice $invoice
    )
    {
        if ($this->stripeTax->isEnabled()) {
            foreach ($invoice->getAllItems() as $item) {
                $item->setInitialTaxAmount($item->getTaxAmount());
                $item->setInitialBaseTaxAmount($item->getBaseTaxAmount());
                $item->setInitialDiscountTaxCompensationAmount($item->getDiscountTaxCompensationAmount());
                $item->setInitialBaseDiscountTaxCompensationAmount($item->getBaseDiscountTaxCompensationAmount());

                $item->setInitialPrice($item->getPrice());
                $item->setInitialBasePrice($item->getBasePrice());
                $item->setInitialPriceInclTax($item->getPriceInclTax());
                $item->setInitialBasePriceInclTax($item->getBasePriceInclTax());
                $item->setInitialRowTotal($item->getRowTotal());
                $item->setInitialBaseRowTotal($item->getBaseRowTotal());
                $item->setInitialRowTotalInclTax($item->getRowTotalInclTax());
                $item->setInitialBaseRowTotalInclTax($item->getBaseRowTotalInclTax());
            }

            $invoice->setInitialShippingTaxAmount($invoice->getShippingTaxAmount());
            $invoice->setInitialBaseShippingTaxAmount($invoice->getBaseShippingTaxAmount());
            $invoice->setInitialShippingDiscountTaxCompensationAmount($invoice->getShippingDiscountTaxCompensationAmount());
            $invoice->setInitialBaseShippingDiscountTaxCompensationAmnt($invoice->getBaseShippingDiscountTaxCompensationAmnt());
            $invoice->setInitialTaxAmount($invoice->getTaxAmount());
            $invoice->setInitialBaseTaxAmount($invoice->getBaseTaxAmount());
            $invoice->setInitialDiscountTaxCompensationAmount($invoice->getDiscountTaxCompensationAmount());
            $invoice->setInitialBaseDiscountTaxCompensationAmount($invoice->getBaseDiscountTaxCompensationAmount());
            $invoice->setInitialGrandTotal($invoice->getGrandTotal());
            $invoice->setInitialBaseGrandTotal($invoice->getBaseGrandTotal());

            $invoice->setInitialSubtotal($invoice->getSubtotal());
            $invoice->setInitialBaseSubtotal($invoice->getBaseSubtotal());
            $invoice->setInitialSubtotalInclTax($invoice->getSubtotalInclTax());
            $invoice->setInitialBaseSubtotalInclTax($invoice->getBaseSubtotalInclTax());

            $invoice->setInitialShippingAmount($invoice->getShippingAmount());
            $invoice->setInitialBaseShippingAmount($invoice->getBaseShippingAmount());
            $invoice->setInitialShippingInclTax($invoice->getShippingInclTax());
            $invoice->setInitialBaseShippingInclTax($invoice->getBaseShippingInclTax());
        }

        return $result;
    }

    private function setStripeCalculatedValues($invoice)
    {
        $response = $this->stripeTax->getResponse();
        $order = $invoice->getOrder();
        foreach ($invoice->getAllItems() as $item) {
            $orderItem = $item->getOrderItem();
            $orderItemQty = $orderItem->getQtyOrdered();

            if ($orderItemQty) {
                if ($orderItem->isDummy() || $item->getQty() <= 0) {
                    // in case the order item is a bundle dynamic product, add the GW data to the invoice item
                    if ($orderItem->getHasChildren() &&
                        $orderItem->isChildrenCalculated() &&
                        $this->giftOptionsHelper->itemHasGiftOptions($orderItem)
                    ) {
                        $this->setItemGwCalculatedData($response, $item, $invoice);
                    }
                    continue;
                }
                $this->setItemCalculatedData($response, $item, $invoice);
                if ($this->giftOptionsHelper->itemHasGiftOptions($orderItem)) {
                    $this->setItemGwCalculatedData($response, $item, $invoice);
                }
            }
        }

        if ($this->invoiceHelper->canIncludeShipping($invoice)) {
            $shippingItem = $response->getShippingCostData();
            $this->transactionItem->prepareForShipping($invoice, $shippingItem);
            $calculatedShippingValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());
            $this->transactionItem->setUseBaseCurrency(true)
                ->setBaseCurrencyPricesForShipping($invoice);
            $baseCalculatedShippingValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());

            $invoice->setStripeCalculatedShippingValues($calculatedShippingValues);
            $invoice->setStripeBaseCalculatedShippingValues($baseCalculatedShippingValues);
        }

        if ($this->giftOptionsHelper->salesObjectHasGiftOptions($order) &&
            $order->getGwTaxAmount() != $order->getGwTaxAmountInvoiced()
        ) {
            $this->setGwCalculatedData($response, $order, $invoice);
        }

        if ($this->giftOptionsHelper->salesObjectHasPrintedCard($order) &&
            $order->getGwCardTaxAmount() != $order->getGwCardTaxInvoiced()
        ) {
            $this->setPrintedCardCalculatedData($response, $order, $invoice);
        }

        $invoice->setStripeTaxCalculationId($response->getTaxCalculationId());
    }

    private function setItemCalculatedData($response, $item, $invoice)
    {
        $lineItem = $response->getLineItemData($this->lineItemsHelper->getReferenceForInvoiceTax($item, $invoice->getOrder()));
        $this->transactionItem->prepare($item, $lineItem);
        $calculatedValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());
        $this->transactionItem->setUseBaseCurrency(true)
            ->setBaseCurrencyPrices($item);
        $baseCalculatedValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());

        $item->setStripeCalculatedValues($calculatedValues);
        $item->setStripeBaseCalculatedValues($baseCalculatedValues);
    }

    private function setItemGwCalculatedData($response, $item, $invoice)
    {
        $lineItem = $response->getLineItemData($this->giftOptionsHelper->getItemGwReferenceForInvoiceTax($item, $invoice->getOrder()));
        $this->transactionItem->prepareItemGw($item, $lineItem);
        $calculatedValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());
        $this->transactionItem->setUseBaseCurrency(true)
            ->setItemGwBaseCurrencyPrices($item);
        $baseCalculatedValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());

        $item->setStripeItemGwCalculatedValues($calculatedValues);
        $item->setStripeItemGwBaseCalculatedValues($baseCalculatedValues);
    }

    private function setGwCalculatedData($response, $order, $invoice)
    {
        $lineItem = $response->getLineItemData($this->giftOptionsHelper->getSalesObjectGiftOptionsReference($order));
        $this->transactionItem->prepareSalesObjectGW($order, $lineItem);
        $calculatedValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());
        $this->transactionItem->setUseBaseCurrency(true)
            ->setSalesObjectGwBaseCurrencyPrices($order);
        $baseCalculatedValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());

        $invoice->setStripeGwCalculatedValues($calculatedValues);
        $invoice->setStripeGwBaseCalculatedValues($baseCalculatedValues);
    }

    private function setPrintedCardCalculatedData($response, $order, $invoice)
    {
        $lineItem = $response->getLineItemData($this->giftOptionsHelper->getSalesObjectPrintedCardReference($order));
        $this->transactionItem->preparePrintedCard($order, $lineItem);
        $calculatedValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());
        $this->transactionItem->setUseBaseCurrency(true)
            ->setPrintedCardBaseCurrencyPrices($order);
        $baseCalculatedValues = $this->getCalculatedValues($invoice->getBaseToOrderRate(), $invoice->getBaseCurrencyCode());

        $invoice->setStripePrintedCardCalculatedValues($calculatedValues);
        $invoice->setStripePrintedCardBaseCalculatedValues($baseCalculatedValues);
    }

    private function getCalculatedValues($baseToOrderRate, $baseCurrencyCode)
    {
        $stripeValues = $this->taxCalculationHelper->getStripeCalculatedValuesForInvoice(
            $this->transactionItem, $baseToOrderRate, $baseCurrencyCode
        );

        return $this->taxCalculationHelper->calculatePrices(
            $this->transactionItem, $stripeValues, $this->transactionItem->getQuantity()
        );
    }
}